#include "../GUI/MiniGUI.h"
#include "../GUI/Font.h"
#include "../../RecursiveFunctions.h"
#include <functional>
#include <vector>
using namespace std;
using namespace MiniGUI;

namespace {
    /* Range constants for the callbacks. */
    const int kLow  = 0;
    const int kHigh = 10000000;
    const int kStep = 100000;
    static_assert(kLow % kStep == 0,  "Low value must be multiple of step size.");
    static_assert(kHigh % kStep == 0, "High value must be a multiple of step size.");

    /* Total number of data points sampled. */
    const int kNumPoints = 1 + (kHigh - kLow) / kStep;

    /* Line graph constants. */
    const size_t kMaxAxisLabels = 10;    // Max number of major tick marks on the X axis
    const char   kAxisColor[] = "#555555"; // Davy's gray
    const Font   kAxisFont(FontFamily::SERIF, FontStyle::NORMAL, 8, kAxisColor);
    const size_t kNumAxisTicks = 11;

    /* Colors to use for various values of k. */
    const vector<string> kLineColors = {
        "#CC0000", // Rosso Corsa
        "#EE7F2D", // Princeton Orange
        "#FFC40C", // Mikado Yellow
        "#008000", // Office Green
        "#007BA7", // Cerulean
        "#B53389", // Fandango
        "#343434", // Jet
    };

    /* Content areas. */
    const double kHeaderHeight    = 50;
    const Font kHeaderFont(FontFamily::SERIF, FontStyle::BOLD_ITALIC, 24, kAxisColor);

    /* General graphics constants. */
    const string kBackgroundColor = "White";
    const double kPadding = 20;

    /* Legend. */
    /* Legend area, expressed as an offset from the start of the chart area. */
    const double kLegendXOffset   = 100;
    const double kLegendYOffset   =   0;
    const double kLegendWidth     = 150;
    const double kLegendHeight    = 200;
    const Font kLegendFont(FontFamily::SANS_SERIF, FontStyle::NORMAL, 12, kAxisColor);

    /* Function, along with its name. */
    struct NamedFunction {
        string name;
        function<int(int)> callback;
    };

    /* Plots charts of different functions. */
    class GraphGUI: public ProblemHandler {
    public:
        GraphGUI(const vector<NamedFunction>& functions, GWindow& window);

        void settingUp() override;

    protected:
        void repaint() override;

    private:
        /* Passed in in constructor. */
        vector<NamedFunction> functions;

        /* Evaluated in settingUp. */
        vector<vector<int>> results;
        vector<string>      fnNames;

        vector<string> axisLabels;   // Labels to use on the x axis.

        void drawChart (const GRectangle& bounds);
        void drawHeader(const GRectangle& bounds);
        void drawLegend(const GRectangle& bounds);
    };

    GraphGUI::GraphGUI(const vector<NamedFunction>& functions, GWindow& window) :
        ProblemHandler(window),
        functions(functions) {
    }

    void GraphGUI::settingUp() {
        /* Draw a message saying we're calculating things. */
        clearDisplay(window(), kBackgroundColor);
        auto render = TextRender::construct("Calculating...",
                                            { 0, 0, window().getCanvasWidth(), window().getCanvasHeight() },
                                            kHeaderFont);
        render->alignCenterHorizontally();
        render->alignCenterVertically();
        render->draw(window());
        window().repaint();

        /* Run all the callbacks to get the underlying data. */
        for (const auto& fn: functions) {
            fnNames.push_back(fn.name);

            vector<int> values;
            for (int n = kLow; n <= kHigh; n += kStep) {
                values.push_back(fn.callback(n));
            }
            results.push_back(values);
        }

        /* Compute axis labels. */
        for (size_t i = 0; i < kNumAxisTicks; i++) {
            size_t effectiveIndex = kLow + (kHigh / (kNumAxisTicks - 1)) * i;

            string label;
            if (effectiveIndex == 0) {
                label = "n=" + to_string(kLow);
            } else if (effectiveIndex <= kHigh) {
                label = addCommasTo(effectiveIndex);
            } else {
                label = ""; // Not needed, but helps clarify intent.
            }

            axisLabels.push_back(label);
        }
    }

    void GraphGUI::repaint() {
        clearDisplay(window(), kBackgroundColor);

        /* Space for the header. */
        GRectangle header = {
            kPadding, kPadding,
            window().getCanvasWidth() - 2 * kPadding,
            kHeaderHeight
        };

        double chartTop = header.y + header.height;

        /* Set up the chart. */
        GRectangle chart = {
            kPadding, chartTop,
            window().getCanvasWidth()  - 2 * kPadding,
            window().getCanvasHeight() - chartTop - kPadding
        };

        /* And the legend. */
        GRectangle legend = {
            chart.x + kLegendXOffset, chart.y + kLegendYOffset,
            kLegendWidth, kLegendHeight
        };

        drawHeader(header);
        drawChart (chart);
        drawLegend(legend);
    }

    void GraphGUI::drawHeader(const GRectangle& bounds) {
        auto render = TextRender::construct("Function Plot", bounds, kHeaderFont, LineBreak::NO_BREAK_SPACES);
        render->alignCenterHorizontally();
        render->alignCenterVertically();
        render->draw(window());
    }

    void GraphGUI::drawChart(const GRectangle& bounds) {
        /* Y-axis labels will be empty. We need a minimum of two, though, so we'll make those. */
        vector<string> yLabels(2);

        /* Convert our results into line segments. */

        /* Find the maximum value across everything. */
        double maxValue = 0;
        double minValue = numeric_limits<double>::infinity();
        for (const auto& result: results) {
            for (double val: result) {
                maxValue = max(val, maxValue);
                minValue = min(val, minValue);
            }
        }

        /* Nudge the max value up a bit to avoid dividing by zero. */
        maxValue = nextafter(maxValue, numeric_limits<double>::infinity());

        /* Normalize everything by that amount. */
        vector<vector<GPoint>> lines;
        for (const auto& result: results) {
            vector<GPoint> line;
            for (int i = 0; i < result.size(); i++) {
                line.push_back({ i * 1.0 / (kNumPoints - 1), (result[i] - minValue) / (maxValue - minValue) });
            }
            lines.push_back(line);
        }

        /* Draw it! */
        LineGraphRender::construct(lines,
                                   axisLabels,
                                   yLabels,
                                   kNumAxisTicks,
                                   0,
                                   bounds,
                                   kAxisFont,
                                   kAxisFont,
                                   kLineColors,
                                   kAxisColor)->draw(window());
    }

    void GraphGUI::drawLegend(const GRectangle& bounds) {
        /* Set up the legend. Don't draw it yet; we need to clear the area behind it. */
        auto legend = LegendRender::construct(fnNames, kLineColors, bounds, kLegendFont, kLegendFont.color());
        window().setColor(kBackgroundColor);
        window().fillRect(legend->computedBounds());

        /* Now draw it. */
        legend->draw(window());
    }
}

GRAPHICS_HANDLER("Plot compute_h", GWindow& window) {
    vector<NamedFunction> fns = {
        { "h(n)", compute_h },
        { "10n",  [](int n) { return 10 * n; } }
    };

    return make_shared<GraphGUI>(fns, window);
}
